<?php
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// CORS headers
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json');

// Handle preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Include database connection
require_once 'db_connect.php';

try {
    // Get POST data
    $sender_id = $_POST['sender_id'] ?? null;
    $reciver_id = $_POST['reciver_id'] ?? null;

    error_log("sendrequest.php - sender_id: $sender_id, reciver_id: $reciver_id");

    if (!$sender_id || !$reciver_id) {
        http_response_code(400);
        echo json_encode([
            "status" => "error", 
            "message" => "Missing params"
        ]);
        exit;
    }

    // Validate numeric
    if (!is_numeric($sender_id) || !is_numeric($reciver_id)) {
        http_response_code(400);
        echo json_encode([
            "status" => "error", 
            "message" => "Invalid params"
        ]);
        exit;
    }

    // Prevent sending request to self
    if ($sender_id == $reciver_id) {
        echo json_encode([
            "status" => "error",
            "message" => "Cannot send request to yourself"
        ]);
        exit;
    }

    // STEP 1: Delete any rejected requests between these users
    // This allows users to send new requests after rejection
    $deleteRejected = $conn->prepare("
        DELETE FROM user_requests
        WHERE ((sender_id = ? AND reciver_id = ?)
           OR (sender_id = ? AND reciver_id = ?))
        AND request_status = 'Rejected'
    ");
    
    if ($deleteRejected) {
        $deleteRejected->bind_param("iiii", $sender_id, $reciver_id, $reciver_id, $sender_id);
        $deleteRejected->execute();
        $deleted_count = $deleteRejected->affected_rows;
        if ($deleted_count > 0) {
            error_log("sendrequest.php - Deleted $deleted_count rejected request(s) between users $sender_id and $reciver_id");
        }
        $deleteRejected->close();
    }

    // STEP 2: Check if ACTIVE request already exists (Pending or Accepted)
    $check = $conn->prepare("
        SELECT id, request_status FROM user_requests
        WHERE ((sender_id = ? AND reciver_id = ?)
           OR (sender_id = ? AND reciver_id = ?))
        AND request_status IN ('Pending', 'Accepted')
    ");
    
    if (!$check) {
        throw new Exception("Database prepare failed: " . $conn->error);
    }

    $check->bind_param("iiii", $sender_id, $reciver_id, $reciver_id, $sender_id);
    $check->execute();
    $result = $check->get_result();

    if ($result->num_rows > 0) {
        $row = $result->fetch_assoc();
        echo json_encode([
            "status" => "exists", 
            "message" => "Request already exists",
            "existing_status" => $row['request_status']
        ]);
        $check->close();
        exit;
    }
    $check->close();

    // STEP 3: Insert new request
    $stmt = $conn->prepare("
        INSERT INTO user_requests (sender_id, reciver_id, request_status)
        VALUES (?, ?, 'Pending')
    ");
    
    if (!$stmt) {
        throw new Exception("Database prepare failed: " . $conn->error);
    }

    $stmt->bind_param("ii", $sender_id, $reciver_id);
    
    if ($stmt->execute()) {
        $request_id = $conn->insert_id;
        error_log("sendrequest.php - Successfully created request ID: $request_id");
        
        echo json_encode([
            "status" => "success", 
            "message" => "Request sent",
            "request_id" => $request_id
        ]);
    } else {
        throw new Exception("Insert failed: " . $stmt->error);
    }
    
    $stmt->close();
    $conn->close();

} catch (Exception $e) {
    error_log("sendrequest.php Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        "status" => "error", 
        "message" => "Internal server error",
        "debug" => $e->getMessage() // Remove this line in production
    ]);
}
?>